//------------------------------------------------------------------------------------------------
// Texture samplers
//------------------------------------------------------------------------------------------------

//@:-texformat mipColor_tex RGBA8
//
sampler2D CoC_tex					: register(s1);	// CoC precomputed values for DOF
sampler2D depth_tex					: register(s2);	// depth tex
sampler2D mipColor_tex				: register(s4);	// mipmapped color tex, starting from what would be mip1 of fullColor_tex

//------------------------------------------------------------------------------------------------
// Shader constants
//------------------------------------------------------------------------------------------------

float4x4 motion_mtx         : register(c132);	// projects fragment from texture clip-space to texture clip space of the previous frame
float3 dofParams			: register(c4);		// { dofMaxMipmapLevel, dofCoCTexLod, HDRrange }
float4 CoCParams			: register(c5);		// { CoCScale, CoCBias, CoCMipMapLevelFactor, 1.0 / textureSize }

//------------------------------------------------------------------------------------------------
// Depth sampling functions
//------------------------------------------------------------------------------------------------

inline float unpackRGBDepth(float3 rgbDepth)
{
	return dot((float3)round(rgbDepth * 255.0), float3(65536.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0));
}


inline float readDepthTextureExt(float2 uv, inout float3 d)
{
	#if defined(CG_PS3) || defined(_Z_ARG)
		d = tex2D(depth_tex, uv).arg;
		return unpackRGBDepth(d);
	#elif defined(_WINPC) && defined(_Z_RGB)
		d = tex2D(depth_tex, uv).rgb;
		return unpackRGBDepth(d);
	#else
		#if defined(USE_INVERTED_PROJECTION_MTX)
			float r = 1.0 - tex2D(depth_tex, uv).r;
		#else
			float r = tex2D(depth_tex, uv).r;
		#endif
		float iDepth = 256.0 * modf(r * 16777215.0/65536.0, d.x);
		d.z = modf(iDepth, d.y);
		d.xy /= 255.0;
		return r;
	#endif
}


inline float readDepthTexture(float2 uv)
{
	#if defined(CG_PS3) || defined(_Z_ARG)
		float3 d = tex2D(depth_tex, uv).arg;
		return unpackRGBDepth(d);
	#elif defined(_WINPC) && defined(_Z_RGB)
		float3 d = tex2D(depth_tex, uv).rgb;
		return unpackRGBDepth(d);
	#else
		#if defined(USE_INVERTED_PROJECTION_MTX)
			float r = 1.0 - tex2D(depth_tex, uv).r;
		#else
			float r = tex2D(depth_tex, uv).r;
		#endif
		return r;
	#endif
}


//------------------------------------------------------------------------------------------------
// Circle of Confusion
//------------------------------------------------------------------------------------------------
#ifdef COC


// Compute a CoC (Circle Of Confusion) value from a depth value in the depth buffer.
float4 computecoc_px( VS_OUTMAINFILTER In ) : COLOR
{
	const float CoCScale    = In.CoCParams.x;
	const float CoCBias     = In.CoCParams.y;
	const float maxMipLevel = In.CoCParams.z;

	// Get the raw non-linear depth
	float3 HiMedLoDepth;
	float rawDepth = readDepthTextureExt(In.texcoord.xy, HiMedLoDepth);

	// CoC computation (signed value):v typically a value ranging in [-8.0, 8.0]
	float CoC = CoCScale * rawDepth + CoCBias;
	float thisLOD = saturate(abs(CoC) / maxMipLevel);

	// Pack a 24-bit depth representation into xyz and the current LOD into w.
	return float4(HiMedLoDepth, thisLOD);
}

//_________________________________________________________________________________________________


float4 computecocfar_px( VS_OUTMAINFILTER In ) : COLOR
{
	const float CoCScale    = In.CoCParams.x;
	const float CoCBias     = In.CoCParams.y;
	const float maxMipLevel = In.CoCParams.z;

	// Get the raw non-linear depth
	float rawDepth = readDepthTexture(In.texcoord.xy);

	// CoC computation (signed value):v typically a value ranging in [-8.0, 8.0]
	float CoC = CoCScale * rawDepth + CoCBias;
	float thisLOD = saturate(CoC / maxMipLevel);
	return float4(0.0, 0.0, 0.0, thisLOD);
}

//_________________________________________________________________________________________________

/**
 * Compute a CoC (Circle Of Confusion) value from a depth value in the depth buffer.
 *
 * The fragment also computes the vector of motion, but in texture clip space.
 * Texture clip space is derived from clip space, but where xyz have been scaled to [0; 1].
 * Furthermore, y has been complemented by 1 (y' = 1 - y), which comes from the sampling of the depth-buffer
 * when using D3D convention.
 * The vector of motion is finally scaled from [-0.125; 0.125] to [0; 1]. This last step is necessary
 * to save accuracy as each component will be stored as 1 byte. Since it is texture clip space,
 * vectors of motion that are wider than 1/8 of the screen dimension are implicitely clamped.
 */
float4 computecocmotion_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	const float CoCScale    = CoCParams.x;
	const float CoCBias     = CoCParams.y;
	const float maxMipLevel = CoCParams.z;

	// z fragment in PPS
	float zFragPP = readDepthTexture(In.uvSet01.xy);

	// CoC computation (signed value):v typically a value ranging in [-8.0, 8.0]
	float CoC = CoCScale * zFragPP + CoCBias;
	float thisLOD = saturate(abs(CoC) / maxMipLevel);

	// frag coordinate in texture clip space and with complement y' coord (y' = 1 - y),
	// and x in [0;1], y' in [0;1], z in [0;1]
	float4 fragClipT0 = float4(In.uvSet01.zw, zFragPP, 1.0);
	float4 fragClipT1 = mul(fragClipT0, motion_mtx);
	fragClipT1 /= fragClipT1.w;
	
	// compute vector of motion: from t1 to t0
	// and scales it from [-0.2; 0.2] to [0, 1] to save it in a texture
	return float4((fragClipT0.xy - fragClipT1.xy + 0.2) / 0.4, zFragPP, thisLOD);
}

//_________________________________________________________________________________________________

float4 computecocfarmotion_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	const float CoCScale    = CoCParams.x;
	const float CoCBias     = CoCParams.y;
	const float maxMipLevel = CoCParams.z;

	// z fragment in PPS
	float zFragPP = readDepthTexture(In.uvSet01.xy);

	// CoC computation (signed value):v typically a value ranging in [-8.0, 8.0]
	float CoC = CoCScale * zFragPP + CoCBias;
	float thisLOD = saturate(CoC / maxMipLevel);

	// frag coordinate in texture clip space and with complement y' coord (y' = 1 - y),
	// and x in [0;1], y' in [0;1], z in [0;1]
	float4 fragClipT0 = float4(In.uvSet01.zw, zFragPP, 1.0);
	float4 fragClipT1 = mul(fragClipT0, motion_mtx);
	fragClipT1 /= fragClipT1.w;
	
	// compute vector of motion: from t1 to t0
	// and scales it from [-0.2; 0.2] to [0, 1] to save it in a texture
	return float4((fragClipT0.xy - fragClipT1.xy + 0.2) / 0.4, zFragPP, thisLOD);
}

//_________________________________________________________________________________________________

float4 computemotion_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	// z fragment in PPS
	float zFragPP = tex2D(depth_tex, In.uvSet01.xy).r;
	
	// frag coordinate in texture clip space and with complement y' coord (y' = 1 - y),
	// and x in [0;1], y' in [0;1], z in [0;1]
	float4 fragClipT0 = float4(In.uvSet01.zw, zFragPP, 1.0);
	float4 fragClipT1 = mul(fragClipT0, motion_mtx);
	fragClipT1 /= fragClipT1.w;

	// compute vector of motion: from t1 to t0
	// and scales it from [-0.2; 0.2] to [0, 1] to save it in a texture
	return float4((fragClipT0.xy - fragClipT1.xy + 0.2) / 0.4, 0.0, 0.0);
}
#endif

//------------------------------------------------------------------------------------------------
// Depth of Field
//------------------------------------------------------------------------------------------------

struct VS_OUTDOFBLEND
{
	float4 position  : POSITION;
	float4 texCoord  : TEXCOORD0;
	float2 dofParams : TEXCOORD1;
};

VS_OUTDOFBLEND blenddofmips_vx(VS_IN_2D input)
{
	VS_OUTDOFBLEND output;
	
	output.position = input.position;
#ifdef _WINPC
	float2 tc = input.uvSet0;
#else
	float2 tc = input.position.xy * float2(0.5, -0.5) + 0.5;
#endif
	output.texCoord = float4(tc * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy, 0.0, 0.0);
	output.dofParams = dofParams.xy;
	return output;
}

#if defined(DOF_FAR)

float4 blenddofmipsfar_px( VS_OUTDOFBLEND In ) : COLOR
{
	// Calculate the LOD to use.
	float thisLOD = tex2Dlod(CoC_tex, In.texCoord).a * In.dofParams.x;

	// Recover a defocussed texel using trilinear interpolation of the blurred mipchain.
	float3 blurColour = tex2Dlod(mipColor_tex, float4(In.texCoord.xyz, thisLOD - 1.0)).rgb;

	// Return this colour with an alpha blend for low LODs.
	return float4(blurColour, saturate(thisLOD));
}

#else

float4 blenddofmips_px( VS_OUTDOFBLEND In ) : COLOR
{
	// The base mip level contains linear depth in xyz and the raw LOD in w.
	// The lower mip level contains region averaged versions of the above.
	float4 CoCHiDef  = tex2Dlod(CoC_tex, In.texCoord);
	float4 CoCLoDef  = tex2Dlod(CoC_tex, float4(In.texCoord.xyz, In.dofParams.y));

	// Get working data.
	float thisDist = unpackRGBDepth(CoCHiDef.xyz);
	float areaDist = unpackRGBDepth(CoCLoDef.xyz);
	float thisLOD  = CoCHiDef.w;
	float areaLOD  = CoCLoDef.w;

	// Use the original LOD for this fragment, unless it is both more distant than the region
	// average and also sharper than the region average, in which case take the boosted average
	// LOD. This will help foreground blurred shapes to bleed into background regions.
	if ((thisDist > areaDist) && (thisLOD < areaLOD))
		thisLOD = min(2.0 * areaLOD - thisLOD, areaLOD + 0.05);

	// Rescale the LOD.
	thisLOD = thisLOD * In.dofParams.x;

	// Recover a defocussed texel using trilinear interpolation of the blurred mipchain.
	float3 blurColour = tex2Dlod(mipColor_tex, float4(In.texCoord.xyz, thisLOD - 1.0)).rgb;

	// Return this colour with an alpha blend for low LODs.
	return float4(blurColour, saturate(thisLOD));
}

#endif
